import React, { useState, useEffect, useRef } from "react";
import { generateUUID } from "./lib/utils";
import { Play, Square, Settings as SettingsIcon, Download, Upload, Plus, FileText, PauseCircle, MousePointerClick, CheckSquare, XSquare, Undo2, Redo2, Sparkles } from "lucide-react";
import { DataTable } from "./components/DataTable";
import { Settings } from "./components/Settings";
import { Logs } from "./components/Logs";
import { SheetTabs } from "./components/SheetTabs";
import * as api from "./lib/api";
import { callAdvancedGeminiApi } from "./lib/gemini";
import {
    generateIntro,
    generateContent,
    generateFAQs,
    generateMeta,
    generateTitle
} from "./lib/prompts";
import { generateCSV, parseCSV } from "./lib/csv";
import { COLS } from "./components/DataTable";
import { useHistory } from "./hooks/useHistory";
import { useAuth } from "./context/AuthContext";
import logo from "./image/Digital-Rackup-Logo.png";

export default function Dashboard() {
    // History Hook for Sheets (instead of just tasks)
    const {
        state: sheets,
        set: setSheets,
        undo,
        redo,
        canUndo,
        canRedo,
        reset: resetSheets,
        setSilent: setSheetsSilent
    } = useHistory([]);

    const [activeSheetId, setActiveSheetId] = useState(null);

    // Auth
    const { logout, user } = useAuth();

    const [selectedTasks, setSelectedTasks] = useState([]);
    const [rangeStart, setRangeStart] = useState("");
    const [rangeEnd, setRangeEnd] = useState("");

    const [appState, setAppState] = useState({
        isProcessing: false,
        status: 'idle', // idle, running, paused, stopping
        progress: 0,
        currentTask: ''
    });
    const [logs, setLogs] = useState([]);
    const [settings, setSettings] = useState({
        apiKeys: [],
        provider: 'gemini',
        longcatApiKey: '',
        model: 'gemini-flash-latest',
        temperature: 0.7
    });
    const [showSettings, setShowSettings] = useState(false);

    // Refs
    const isProcessingRef = useRef(false);
    const statusRef = useRef('idle'); // idle, running, paused
    const pauseResolveRef = useRef(null);
    const settingsRef = useRef(settings);
    const abortControllerRef = useRef(null);
    const sheetsRef = useRef(sheets); // Ref for async access (processing loop)

    // Sync refs
    useEffect(() => { settingsRef.current = settings; }, [settings]);
    useEffect(() => { sheetsRef.current = sheets; }, [sheets]);

    useEffect(() => {
        loadInitialData();
    }, []);

    const loadInitialData = async () => {
        try {
            // Try loading sheets first
            let initialSheets = [];
            try {
                initialSheets = await api.getSheets();
            } catch (err) {
                console.warn("Failed to fetch sheets, attempting legacy migration...", err);
            }

            // Migration: If no sheets found (or API failed), try legacy tasks
            if (!initialSheets || initialSheets.length === 0) {
                try {
                    const legacyTasks = await api.getTasks();
                    if (legacyTasks && legacyTasks.length > 0) {
                        const defaultSheet = {
                            id: generateUUID(),
                            name: "Sheet 1",
                            tasks: legacyTasks
                        };
                        initialSheets = [defaultSheet];
                        // Try to persist the migration immediately
                        await api.saveSheets(initialSheets).catch(e => console.error("Failed to save migrated sheets:", e));
                    } else {
                        // Totally new user or empty
                        const defaultSheet = {
                            id: generateUUID(),
                            name: "Sheet 1",
                            tasks: []
                        };
                        initialSheets = [defaultSheet];
                    }
                } catch (legacyErr) {
                    console.error("Failed to fetch legacy tasks:", legacyErr);
                    // Absolute fallback if everything fails
                    initialSheets = [{ id: generateUUID(), name: "Sheet 1", tasks: [] }];
                }
            }

            resetSheets(initialSheets);
            if (initialSheets.length > 0) {
                setActiveSheetId(initialSheets[0].id);
            }

            try {
                const savedSettings = await api.getSettings();
                if (savedSettings && Object.keys(savedSettings).length > 0) setSettings(prev => ({ ...prev, ...savedSettings }));
            } catch (e) { console.error("Failed to load settings:", e); }

        } catch (error) {
            console.error("Critical error in loadInitialData:", error);
            const defaultSheet = { id: generateUUID(), name: "Sheet 1", tasks: [] };
            resetSheets([defaultSheet]);
            setActiveSheetId(defaultSheet.id);
        }
    };

    // Helper to get active sheet
    const activeSheet = sheets.find(s => s.id === activeSheetId) || { tasks: [] };
    const tasks = activeSheet.tasks || [];

    const addLog = (message, level = 'info') => {
        const timestamp = new Date().toLocaleTimeString();
        setLogs(prev => [...prev, { timestamp, message, level }]);
        api.log(level, message);
    };

    // --- Sheet Operations ---

    const handleAddSheet = () => {
        const newSheet = {
            id: generateUUID(),
            name: `Sheet ${sheets.length + 1}`,
            tasks: []
        };
        const updated = [...sheets, newSheet];
        setSheets(updated);
        setActiveSheetId(newSheet.id);
        api.saveSheets(updated);
    };

    const handleSwitchSheet = (id) => {
        setActiveSheetId(id);
        setSelectedTasks([]); // Clear selection when switching
    };

    const handleRenameSheet = (id, newName) => {
        const updated = sheets.map(s => s.id === id ? { ...s, name: newName } : s);
        setSheets(updated);
        api.saveSheets(updated);
    };

    const handleDeleteSheet = (id) => {
        if (sheets.length <= 1) return; // Prevent deleting last sheet
        const updated = sheets.filter(s => s.id !== id);
        setSheets(updated);
        api.saveSheets(updated);
        if (activeSheetId === id) {
            setActiveSheetId(updated[0].id);
        }
    };

    const handleDuplicateSheet = (id) => {
        const sheetToDuplicate = sheets.find(s => s.id === id);
        if (!sheetToDuplicate) return;

        const newSheet = {
            ...sheetToDuplicate,
            id: generateUUID(),
            name: `${sheetToDuplicate.name} Copy`,
            tasks: JSON.parse(JSON.stringify(sheetToDuplicate.tasks)) // Deep copy
        };

        // Insert after current sheet
        const index = sheets.findIndex(s => s.id === id);
        const updated = [...sheets];
        updated.splice(index + 1, 0, newSheet); // Insert after original

        setSheets(updated);
        setActiveSheetId(newSheet.id);
        api.saveSheets(updated);
    };

    const handleReorderSheets = (newOrder) => {
        setSheets(newOrder);
        api.saveSheets(newOrder); // Persist order
    };

    const handleSheetColorChange = (id, color) => {
        const updated = sheets.map(s => s.id === id ? { ...s, color } : s);
        setSheets(updated);
        api.saveSheets(updated);
    };


    // --- Task Operations (Context: Active Sheet) ---

    const updateActiveSheetTasks = (newTasks, silent = false) => {
        const updatedSheets = sheets.map(s => s.id === activeSheetId ? { ...s, tasks: newTasks } : s);
        if (silent) {
            setSheetsSilent(updatedSheets);
        } else {
            setSheets(updatedSheets);
        }
        api.saveSheets(updatedSheets);
    };

    const addTask = (count = 1) => {
        const newTasks = Array.from({ length: count }).map(() => ({
            id: generateUUID(),
            status: 'idle',
            keyword: '',
            secondaryKeyword: '',
            nlpOutline: '',
            results: {}
        }));

        const updatedTasks = [...tasks, ...newTasks];
        updateActiveSheetTasks(updatedTasks);
    };

    const updateTask = (id, updates) => {
        const updatedTasks = tasks.map(t => t.id === id ? { ...t, ...updates } : t);
        updateActiveSheetTasks(updatedTasks);
    };

    const bulkUpdateTasks = (updates) => {
        const updateMap = new Map();
        updates.forEach(u => {
            const existing = updateMap.get(u.id) || {};
            updateMap.set(u.id, { ...existing, ...u.changes });
        });

        const updatedTasks = tasks.map(t => {
            if (updateMap.has(t.id)) {
                return { ...t, ...updateMap.get(t.id) };
            }
            return t;
        });

        updateActiveSheetTasks(updatedTasks);
    };

    const deleteTask = (id) => {
        const updatedTasks = tasks.filter(t => t.id !== id);
        updateActiveSheetTasks(updatedTasks);
        // api.deleteTask(id); // With sheets, we save the whole sheet structure, so individual delete might be redundant if we saveSheets
    };

    const toggleSelect = (id, isChecked) => {
        if (id === 'all') {
            setSelectedTasks(isChecked ? tasks.map(t => t.id) : []);
        } else {
            setSelectedTasks(prev => isChecked ? [...prev, id] : prev.filter(tid => tid !== id));
        }
    };

    // --- Processing Logic ---

    const updateTaskSilent = (taskId, updates) => {
        // This is tricky with sheets structure and async loop
        // We need to update the specific task in the specific sheet within the sheets array
        // But inside loop we use functional updates or ref?
        // Let's use functional update on setSheetsSilent
        setSheetsSilent(prevSheets => prevSheets.map(s => {
            if (s.id === activeSheetId) {
                return {
                    ...s,
                    tasks: s.tasks.map(t => t.id === taskId ? { ...t, ...updates } : t)
                };
            }
            return s;
        }));
    };

    const saveCurrentSheetsSilent = (currentSheets) => {
        // This is a helper to persist without triggering history
        // But inside the loop we might use the state coming from setSheetsSilent
        // Actually, we should just save the ref content?
        // api.saveSheets(currentSheets); 
        // Be careful with race conditions.
    };


    const handleProcess = async () => {
        if (statusRef.current === 'running' || statusRef.current === 'paused') return;

        if (selectedTasks.length === 0) return alert("Please select tasks to process.");

        const processingSheetId = activeSheetId; // Capture current sheet ID

        // Start processing
        isProcessingRef.current = true;
        statusRef.current = 'running';
        abortControllerRef.current = new AbortController();
        setAppState(prev => ({ ...prev, isProcessing: true, status: 'running', progress: 0, currentTask: 'Initializing...' }));
        addLog(`🚀 Starting generation for ${selectedTasks.length} tasks`, 'info');

        const taskIdsToProcess = selectedTasks;
        let completedCount = 0;
        let i = 0;

        while (i < taskIdsToProcess.length) {
            if (!isProcessingRef.current) {
                addLog('🛑 Processing Stopped by User.', 'warning');
                break;
            }

            if (statusRef.current === 'paused') {
                setAppState(prev => ({ ...prev, currentTask: 'Paused - Click Resume to continue' }));
                await new Promise(resolve => { pauseResolveRef.current = resolve; });
                statusRef.current = 'running';
                setAppState(prev => ({ ...prev, status: 'running' }));
                if (!isProcessingRef.current) break;
            }

            const taskId = taskIdsToProcess[i];

            // Find task in CURRENT REF
            const currentSheet = sheetsRef.current.find(s => s.id === processingSheetId);
            if (!currentSheet) { i++; continue; } // Sheet deleted?
            const task = currentSheet.tasks.find(t => t.id === taskId);


            if (!task) { i++; continue; }

            setAppState(prev => ({
                ...prev,
                currentTask: `Processing: ${task.keyword}`,
                progress: Math.round((completedCount / taskIdsToProcess.length) * 100)
            }));

            // Silent update status
            setSheetsSilent(prev => prev.map(s => s.id === processingSheetId ? { ...s, tasks: s.tasks.map(t => t.id === taskId ? { ...t, status: 'processing' } : t) } : s));
            addLog(`📍 START TASK: "${task.keyword}"`, 'info');

            let retryCount = 0;
            let success = false;
            let skipTask = false;

            while (!success && !skipTask && isProcessingRef.current) {
                if (statusRef.current === 'paused') {
                    setAppState(prev => ({ ...prev, currentTask: 'Paused - Click Resume to continue' }));
                    await new Promise(resolve => { pauseResolveRef.current = resolve; });
                    statusRef.current = 'running';
                    setAppState(prev => ({ ...prev, status: 'running' }));
                    if (!isProcessingRef.current) break;
                }

                try {
                    // Pass the whole settings object, not just keys
                    const currentSettings = settingsRef.current;
                    const results = {};
                    let title = '';

                    const updateResult = (key, value) => {
                        setSheetsSilent(prev => prev.map(s => s.id === processingSheetId ? { ...s, tasks: s.tasks.map(t => t.id === taskId ? { ...t, results: { ...t.results, [key]: value } } : t) } : s));
                    };


                    addLog(`  ➤ Generating Title...`, 'info');
                    title = await generateTitle(task, currentSettings, addLog, abortControllerRef.current.signal);
                    results.title = title;
                    updateResult('title', title);
                    await new Promise(r => setTimeout(r, 3000));

                    addLog(`  ➤ Generating Intro...`, 'info');
                    results.intro = await generateIntro(task, title, currentSettings, addLog, abortControllerRef.current.signal);
                    updateResult('intro', results.intro);
                    await new Promise(r => setTimeout(r, 3000));

                    addLog(`  ➤ Generating Content...`, 'info');
                    results.content = await generateContent(task, results.intro, title, currentSettings, addLog, abortControllerRef.current.signal);
                    updateResult('content', results.content);
                    await new Promise(r => setTimeout(r, 3000));

                    addLog(`  ➤ Generating FAQs...`, 'info');
                    results.faqs = await generateFAQs(task, results.content, currentSettings, addLog, abortControllerRef.current.signal);
                    updateResult('faqs', results.faqs);
                    await new Promise(r => setTimeout(r, 3000));

                    addLog(`  ➤ Generating Meta Description...`, 'info');
                    results.meta = await generateMeta(task, results.content, title, currentSettings, addLog, abortControllerRef.current.signal);
                    updateResult('meta', results.meta);
                    await new Promise(r => setTimeout(r, 3000));

                    setSheetsSilent(prev => prev.map(s => s.id === processingSheetId ? { ...s, tasks: s.tasks.map(t => t.id === taskId ? { ...t, status: 'completed', results: { ...t.results, ...results } } : t) } : s));

                    addLog(`✅ COMPLETED: "${task.keyword}"`, 'success');
                    // Persist intermediate state?
                    api.saveSheets(sheetsRef.current); // Use ref to get latest
                    success = true;

                } catch (error) {
                    if (error.message === 'Aborted by user') {
                        addLog('🛑 Generation Stopped.', 'warning');
                        skipTask = true;
                    } else if (error.message === 'ALL_KEYS_EXHAUSTED') {
                        retryCount++;
                        if (retryCount === 1) {
                            addLog("⚠️ Rate Limit (All Keys). Waiting 1 minute...", "warning");
                            await new Promise(r => setTimeout(r, 60000));
                        } else if (retryCount === 2) {
                            addLog("⚠️ Rate Limit (All Keys). Waiting 2 minutes...", "warning");
                            await new Promise(r => setTimeout(r, 120000));
                        } else {
                            addLog("⛔ Rate Limit Persistence. PAUSING Process. Please change API Key.", "error");
                            handlePause();
                            await new Promise(resolve => { pauseResolveRef.current = resolve; });
                            statusRef.current = 'running';
                            setAppState(prev => ({ ...prev, status: 'running' }));
                            retryCount = 0;
                        }
                    } else {
                        setSheetsSilent(prev => prev.map(s => s.id === processingSheetId ? { ...s, tasks: s.tasks.map(t => t.id === taskId ? { ...t, status: 'failed', error: error.message } : t) } : s));
                        addLog(`❌ FAILED: "${task.keyword}" - ${error.message}`, 'error');
                        skipTask = true;
                    }
                }
            }
            completedCount++;
            i++;
        }

        isProcessingRef.current = false;
        statusRef.current = 'idle';
        setAppState(prev => ({ ...prev, isProcessing: false, status: 'idle', progress: 100, currentTask: 'Done!' }));
        addLog(`🎉 All tasks completed!`, 'success');
        api.saveSheets(sheetsRef.current);
    };

    const handlePause = () => {
        statusRef.current = 'paused';
        setAppState(prev => ({ ...prev, status: 'paused' }));
    };

    const handleResume = () => {
        if (pauseResolveRef.current) {
            pauseResolveRef.current();
            pauseResolveRef.current = null;
        }
    };

    const handleStop = () => {
        isProcessingRef.current = false;
        statusRef.current = 'stopping';
        if (abortControllerRef.current) abortControllerRef.current.abort();
        setAppState(prev => ({ ...prev, status: 'stopping' }));
        if (pauseResolveRef.current) {
            pauseResolveRef.current();
            pauseResolveRef.current = null;
        }
    };

    const handleRangeSelect = () => {
        const start = parseInt(rangeStart);
        const end = parseInt(rangeEnd);

        if (isNaN(start) || isNaN(end) || start < 1 || end < start) {
            addLog("⚠️ Invalid range. Please enter valid row numbers (e.g., 1-10)", "warning");
            return;
        }

        const newSelection = [];
        for (let i = start - 1; i < end; i++) {
            if (tasks[i]) newSelection.push(tasks[i].id);
        }

        setSelectedTasks(newSelection);
        addLog(`✅ Selected rows ${start} to ${end} (${newSelection.length} tasks)`, "info");
    };

    const handleKeyDownGlobal = (e) => {
        if ((e.ctrlKey || e.metaKey) && e.key === 'z') {
            e.preventDefault();
            if (e.shiftKey) {
                if (canRedo) {
                    redo();
                    addLog("Redo", "info");
                }
            } else {
                if (canUndo) {
                    undo();
                    addLog("Undo", "info");
                }
            }
        }
    };

    useEffect(() => {
        window.addEventListener('keydown', handleKeyDownGlobal);
        return () => window.removeEventListener('keydown', handleKeyDownGlobal);
    }, [canUndo, canRedo, undo, redo]);

    // Import/Export logic needs update to support sheets?
    // User probably wants to export current sheet logic mostly

    const handleExport = () => {
        // Export current sheet only? Or all? Let's do current sheet for now as it matches "Download CSV" expectation
        const dataStr = JSON.stringify(tasks, null, 2);
        const blob = new Blob([dataStr], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `tasks-${activeSheet.name}-${Date.now()}.json`;
        a.click();
        addLog('📥 Exported current sheet (JSON)', 'info');
    };

    const handleExportCSV = () => {
        const csvContent = generateCSV(tasks, COLS.filter(c => !c.isAction && c.key !== 'select'));
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `tasks-${activeSheet.name}-${Date.now()}.csv`;
        a.click();
        addLog('📥 Exported current sheet (CSV)', 'info');
    };

    const handleImport = (e) => {
        const file = e.target.files[0];
        if (!file) return;
        const reader = new FileReader();
        reader.onload = (ev) => {
            try {
                const imported = JSON.parse(ev.target.result);
                // Validate if it's a sheet array or tasks array?
                // Assume tasks array for now
                const validated = imported.map(t => ({
                    ...t,
                    id: t.id || generateUUID(),
                    status: t.status || 'idle'
                }));

                // Append to current sheet
                const updatedTasks = [...tasks, ...validated];
                updateActiveSheetTasks(updatedTasks);

                addLog(`📤 Imported ${validated.length} tasks (JSON)`, 'success');
            } catch (err) {
                addLog(`❌ Import JSON failed: ${err.message}`, 'error');
            }
        };
        reader.readAsText(file);
        e.target.value = null;
    };

    const handleImportCSV = (e) => {
        const file = e.target.files[0];
        if (!file) return;
        const reader = new FileReader();
        reader.onload = (ev) => {
            try {
                const text = ev.target.result;
                const rows = parseCSV(text);
                if (rows.length < 2) throw new Error("CSV file is empty or missing header");

                const headers = rows[0].map(h => h.toLowerCase().trim());
                const colMap = {};
                COLS.forEach(col => {
                    const idx = headers.findIndex(h => h === col.label.toLowerCase() || h === col.key.toLowerCase());
                    if (idx !== -1) colMap[col.key] = idx;
                });

                const newTasks = rows.slice(1).map(row => {
                    const task = {
                        id: generateUUID(),
                        status: 'idle',
                        keyword: '',
                        secondaryKeyword: '',
                        nlpOutline: '',
                        results: {}
                    };
                    Object.entries(colMap).forEach(([key, idx]) => {
                        const val = row[idx];
                        if (val === undefined) return;
                        const col = COLS.find(c => c.key === key);
                        if (col.isResult) {
                            const resKey = col.key === 'meta' ? 'meta' : col.key;
                            task.results[resKey] = val;
                        } else {
                            task[key] = val;
                        }
                    });
                    return task;
                }).filter(t => t.keyword || t.secondaryKeyword || (t.results && Object.keys(t.results).length > 0));

                if (newTasks.length > 0) {
                    const updated = [...tasks, ...newTasks];
                    updateActiveSheetTasks(updated);
                    addLog(`📤 Imported ${newTasks.length} tasks (CSV)`, 'success');
                } else {
                    addLog('⚠️ No valid tasks found in CSV', 'warning');
                }
            } catch (err) {
                addLog(`❌ Import CSV failed: ${err.message}`, 'error');
            }
        };
        reader.readAsText(file);
        e.target.value = null;
    };


    const handleSaveSettings = (newSettings) => {
        setSettings(newSettings);
        api.saveSettings(newSettings);
        setShowSettings(false);
        addLog('⚙️ Settings saved', 'success');
    };

    // Profile State
    const [showProfileModal, setShowProfileModal] = useState(false);
    const [profileName, setProfileName] = useState(user?.displayName || "");
    const [profilePic, setProfilePic] = useState(user?.photoURL || "");
    const [resetEmailSent, setResetEmailSent] = useState(false);

    useEffect(() => {
        if (user) {
            setProfileName(user.displayName || "");
            setProfilePic(user.photoURL || "");
        }
    }, [user]);

    const handleUpdateProfile = async (e) => {
        e.preventDefault();
        try {
            await user.updateUser(profileName, profilePic);
            addLog("✅ Profile updated successfully", "success");
            setShowProfileModal(false);
        } catch (error) {
            addLog(`❌ Profile update failed: ${error.message}`, "error");
        }
    };

    const handleResetPassword = async () => {
        if (!user?.email) return;
        try {
            await user.resetPassword(user.email);
            setResetEmailSent(true);
            addLog(`📧 Password reset email sent to ${user.email}`, "success");
        } catch (error) {
            addLog(`❌ Failed to send reset email: ${error.message}`, "error");
        }
    };

    return (
        <div className="app h-screen flex flex-col overflow-hidden">
            {/* Header */}
            <div className="header glass-panel m-4 mb-0 flex justify-between items-center p-4">
                <div className="header-left flex items-center gap-3">
                    <img src={logo} alt="Logo" className="header-logo" onError={(e) => e.target.style.display = 'none'} />
                    <h1 className="header-title">
                        Content Generator
                    </h1>
                </div>
                <div className="header-right flex items-center gap-4">
                    {/* Profile Section */}
                    <div
                        className="profile-section group"
                        onClick={() => setShowProfileModal(true)}
                        title="Edit Profile"
                    >
                        <div className="profile-image-container">
                            {user?.photoURL ? (
                                <img
                                    src={user.photoURL}
                                    alt="Profile"
                                    className="profile-img"
                                />
                            ) : (
                                <div className="profile-placeholder">
                                    {user?.displayName ? user.displayName.charAt(0).toUpperCase() : (user?.email?.charAt(0).toUpperCase() || 'U')}
                                </div>
                            )}
                            <div className="profile-status-indicator"></div>
                        </div>
                        <span className="profile-name">
                            {user?.displayName || "User"}
                        </span>
                    </div>

                    <button onClick={logout} className="glass-button-secondary text-xs py-1 px-3 h-8">
                        Sign Out
                    </button>

                    <div className="h-8 w-[1px] bg-[var(--border-color)] mx-2"></div>


                    <button onClick={() => addTask(1)} className="btn-secondary hidden sm:flex">
                        <Plus size={16} /> Add Row
                    </button>
                    {/* Compact buttons for mobile */}
                    <button onClick={() => addTask(1)} className="btn-secondary sm:hidden" title="Add Row">
                        <Plus size={16} />
                    </button>

                    <label className="btn-secondary hidden sm:flex" title="Import JSON">
                        <Upload size={16} /> Import JSON
                        <input type="file" accept=".json" onChange={handleImport} style={{ display: 'none' }} />
                    </label>
                    <label className="btn-secondary hidden sm:flex" title="Import CSV/Excel">
                        <Upload size={16} /> Import CSV
                        <input type="file" accept=".csv" onChange={handleImportCSV} style={{ display: 'none' }} />
                    </label>
                    <button onClick={handleExport} className="btn-secondary hidden sm:flex" title="Export JSON">
                        <Download size={16} /> Export JSON
                    </button>
                    <button onClick={handleExportCSV} className="btn-secondary hidden sm:flex" title="Export CSV (Excel)">
                        <Download size={16} /> Export CSV
                    </button>
                    <button onClick={() => setShowSettings(true)} className="btn-secondary">
                        <SettingsIcon size={16} />
                    </button>

                </div>
            </div>

            {/* Controls */}
            <div className="controls">
                <div className="flex gap-2">
                    {appState.status === 'running' && (
                        <button onClick={handlePause} className="btn-secondary" title="Pause">
                            <PauseCircle size={16} /> Pause
                        </button>
                    )}
                    {appState.status === 'paused' && (
                        <button onClick={handleResume} className="btn-success" title="Resume">
                            <Play size={16} /> Resume
                        </button>
                    )}
                    <button
                        onClick={appState.isProcessing ? handleStop : handleProcess}
                        className={appState.isProcessing ? "btn-danger" : "btn-primary"}
                        disabled={selectedTasks.length === 0 && !appState.isProcessing}
                    >
                        {appState.isProcessing ? <><Square size={16} /> Stop</> : <><Play size={16} /> Start</>}
                    </button>
                </div>
                {appState.isProcessing && (
                    <div className="progress-info">
                        <div className="progress-bar">
                            <div className="progress-fill" style={{ width: `${appState.progress}%` }}></div>
                        </div>
                        <span className="progress-text">{appState.currentTask}</span>
                    </div>
                )}
            </div>



            {/* Main Layout */}
            <div className="main-layout p-4 gap-4 bg-[var(--bg-primary)] flex-1 min-h-0 overflow-hidden flex">
                {/* Content Area */}
                <div className="data-table-container glass-panel overflow-hidden flex flex-col">
                    <DataTable
                        tasks={tasks} // Pass tasks from active sheet
                        selectedTasks={selectedTasks}
                        onToggleSelect={toggleSelect}
                        onUpdateTask={updateTask}
                        onBulkUpdateTask={bulkUpdateTasks}
                        onDeleteTask={deleteTask}
                        onAddRow={() => addTask(10)}
                    />
                </div>

                {/* Sidebar */}
                <div className="sidebar w-[350px] flex flex-col gap-4 bg-transparent border-none backdrop-filter-none">

                    {/* Selection Panel (Premium Redesign) */}
                    <div className="premium-panel flex flex-col shrink-0">
                        <div className="premium-header">
                            <div className="premium-title">
                                <Sparkles size={18} className="text-yellow-300 fill-yellow-300" /> Selection
                            </div>
                            <div className="premium-subtitle">
                                {selectedTasks.length} / {tasks.length} selected
                            </div>
                        </div>

                        <div className="p-5 flex flex-col gap-4">
                            {/* Select All / Deselect (Side by Side) */}
                            <div className="flex gap-3">
                                <button
                                    onClick={() => toggleSelect('all', true)}
                                    className="premium-btn flex-1"
                                    title="Select All Rows"
                                >
                                    <CheckSquare size={16} className="text-emerald-400" /> Select All
                                </button>
                                <button
                                    onClick={() => toggleSelect('all', false)}
                                    className="premium-btn flex-1"
                                    title="Deselect All Rows"
                                >
                                    <XSquare size={16} className="text-rose-400" /> Select None
                                </button>
                            </div>

                            {/* Range Select (Row Layout) */}
                            <div className="premium-input-row">
                                <span className="text-xs text-white/50 pl-2 font-medium uppercase tracking-wider">Range</span>
                                <input
                                    className="premium-input"
                                    placeholder="Start"
                                    type="number"
                                    min="1"
                                    value={rangeStart}
                                    onChange={(e) => setRangeStart(e.target.value)}
                                />
                                <span className="text-white/30 font-bold">-</span>
                                <input
                                    className="premium-input"
                                    placeholder="End"
                                    type="number"
                                    min="1"
                                    value={rangeEnd}
                                    onChange={(e) => setRangeEnd(e.target.value)}
                                />
                                <button
                                    onClick={handleRangeSelect}
                                    className="premium-go-btn"
                                >
                                    Go
                                </button>
                            </div>

                            {/* Undo / Redo (Side by Side) */}
                            <div className="flex gap-3">
                                <button
                                    className="premium-btn flex-1"
                                    onClick={undo}
                                    disabled={!canUndo}
                                    title="Undo (Ctrl+Z)"
                                >
                                    <Undo2 size={16} className="text-blue-400" /> Undo
                                </button>
                                <button
                                    className="premium-btn flex-1"
                                    onClick={redo}
                                    disabled={!canRedo}
                                    title="Redo (Ctrl+Shift+Z)"
                                >
                                    <Redo2 size={16} className="text-blue-400" /> Redo
                                </button>
                            </div>
                        </div>
                    </div>

                    {/* Execution Log */}
                    <Logs logs={logs} />
                </div>
            </div>

            {/* Sheet Tabs (Bottom) */}
            <div className="mx-4 mb-2 mt-1 z-10">
                <SheetTabs
                    sheets={sheets}
                    activeSheetId={activeSheetId}
                    onSwitch={handleSwitchSheet}
                    onAdd={handleAddSheet}
                    onRename={handleRenameSheet}
                    onDelete={handleDeleteSheet}
                    onDuplicate={handleDuplicateSheet}
                    onReorder={handleReorderSheets}
                    onColorChange={handleSheetColorChange}
                />
            </div>

            {/* Settings Modal */}
            {showSettings && (
                <Settings
                    settings={settings}
                    onSave={handleSaveSettings}
                    onClose={() => setShowSettings(false)}
                />
            )}

            {/* Profile Modal */}
            {showProfileModal && (
                <div className="modal-overlay" onClick={() => setShowProfileModal(false)}>
                    <div className="modal-content glass-panel" onClick={e => e.stopPropagation()}>
                        <div className="modal-header">
                            <h2>Edit Profile</h2>
                            <button onClick={() => setShowProfileModal(false)} className="text-[var(--text-muted)] hover:text-white">✕</button>
                        </div>
                        <div className="modal-body space-y-4">
                            <form onSubmit={handleUpdateProfile}>
                                <div className="form-group">
                                    <label>Display Name</label>
                                    <input
                                        className="input w-full"
                                        value={profileName}
                                        onChange={e => setProfileName(e.target.value)}
                                        placeholder="Enter your name"
                                    />
                                </div>
                                <div className="form-group">
                                    <label>Profile Picture URL</label>
                                    <input
                                        className="input w-full"
                                        value={profilePic}
                                        onChange={e => setProfilePic(e.target.value)}
                                        placeholder="https://example.com/me.jpg"
                                    />
                                </div>
                                <button type="submit" className="glass-button w-full mt-2">Save Profile</button>
                            </form>

                            <div className="pt-4 border-t border-[var(--border-color)]">
                                <h3 className="text-sm font-bold mb-2">Security</h3>
                                <p className="text-xs text-[var(--text-muted)] mb-2">
                                    Reset password for <strong>{user?.email}</strong>
                                </p>
                                <button
                                    onClick={handleResetPassword}
                                    className="btn-secondary w-full justify-center text-xs"
                                    disabled={resetEmailSent}
                                >
                                    {resetEmailSent ? "Reset Email Sent!" : "Send Password Reset Email"}
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}
